<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Article;
use App\Models\Project;
use App\Models\Service;
use App\Models\Team;
use App\Models\Testimonial;
use App\Models\Message;
use App\Models\JobApplication;
use App\Models\JobListing;
use App\Models\WebsiteVisitor;
use Illuminate\Http\Request;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Display the admin dashboard.
     */
    public function index()
    {
        // Get statistics for dashboard
        $stats = [
            'services' => Service::count(),
            'projects' => Project::count(),
            'articles' => Article::count(),
            'teams' => Team::count(),
            'testimonials' => Testimonial::count(),
            'messages' => Message::where('is_read', false)->count(),
            'job_applications' => JobApplication::count(),
            'active_jobs' => JobListing::where('is_active', true)->count(),
            'total_visitors' => WebsiteVisitor::getTotalVisitors(),
            'today_visitors' => WebsiteVisitor::getTodayVisitors(),
            'month_visitors' => WebsiteVisitor::getThisMonthVisitors(),
        ];

        // Get recent data
        $recent = [
            'messages' => Message::latest()->take(5)->get(),
            'job_applications' => JobApplication::with('jobListing')->latest()->take(5)->get(),
            'articles' => Article::latest()->take(5)->get(),
        ];

        return view('admin.dashboard', compact('stats', 'recent'));
    }

    /**
     * Get visitor chart data via AJAX
     */
    public function getVisitorChartData(Request $request)
    {
        $type = $request->get('type', 'custom');
        
        switch ($type) {
            case '1month':
                $startDate = Carbon::now()->subMonth()->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
            case '2months':
                $startDate = Carbon::now()->subMonths(2)->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
            case '3months':
                $startDate = Carbon::now()->subMonths(3)->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
            case 'custom':
            default:
                $startDate = $request->get('start_date') ? Carbon::parse($request->get('start_date')) : Carbon::now()->subMonth();
                $endDate = $request->get('end_date') ? Carbon::parse($request->get('end_date')) : Carbon::now();
                break;
        }

        $visitorsData = WebsiteVisitor::getVisitorsByDateRange($startDate, $endDate);
        
        $labels = [];
        $data = [];
        
        foreach ($visitorsData as $visitor) {
            $labels[] = Carbon::parse($visitor->visit_date)->format('d M Y');
            $data[] = $visitor->visitors;
        }

        return response()->json([
            'labels' => $labels,
            'data' => $data,
            'total' => array_sum($data),
        ]);
    }
}
