<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\JobApplication;
use App\Models\JobListing;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class JobApplicationController extends Controller
{
    public function index()
    {
        $applications = JobApplication::with('jobListing')->latest()->paginate(10);
        
        // Get stats for dashboard
        $stats = [
            'total' => JobApplication::count(),
            'pending' => JobApplication::where('status', 'pending')->count(),
            'reviewed' => JobApplication::where('status', 'reviewed')->count(),
            'thisMonth' => JobApplication::where('created_at', '>=', now()->startOfMonth())->count(),
        ];
        
        return view('admin.job-applications.index', compact('applications', 'stats'));
    }

    public function create()
    {
        $jobListings = JobListing::where('is_active', true)->orderBy('title')->get();
        return view('admin.job-applications.create', compact('jobListings'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'job_listing_id' => 'required|exists:job_listings,id',
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'cover_letter' => 'required|string',
            'portfolio_url' => 'nullable|url|max:255',
            'resume' => 'nullable|file|mimes:pdf,doc,docx|max:5120', // 5MB max
            'status' => 'nullable|in:pending,reviewed,interview,accepted,rejected',
            'notes' => 'nullable|string',
        ]);

        // Handle file upload
        if ($request->hasFile('resume')) {
            $validated['resume_path'] = $request->file('resume')->store('resumes', 'public');
        }

        // Set default status if not provided
        $validated['status'] = $validated['status'] ?? 'pending';

        JobApplication::create($validated);

        return redirect()->route('admin.job-applications.index')
            ->with('success', 'Lamaran pekerjaan berhasil ditambahkan!');
    }

    public function show(JobApplication $jobApplication)
    {
        $jobApplication->load('jobListing');
        return view('admin.job-applications.show', compact('jobApplication'));
    }

    public function edit(JobApplication $jobApplication)
    {
        $jobListings = JobListing::orderBy('title')->get();
        return view('admin.job-applications.edit', compact('jobApplication', 'jobListings'));
    }

    public function update(Request $request, JobApplication $jobApplication)
    {
        $validated = $request->validate([
            'job_listing_id' => 'required|exists:job_listings,id',
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'cover_letter' => 'required|string',
            'portfolio_url' => 'nullable|url|max:255',
            'resume' => 'nullable|file|mimes:pdf,doc,docx|max:5120', // 5MB max
            'status' => 'required|in:pending,reviewed,interview,accepted,rejected',
            'notes' => 'nullable|string',
            'remove_resume' => 'nullable|boolean',
        ]);

        // Handle resume removal
        if ($request->has('remove_resume') && $jobApplication->resume_path) {
            Storage::disk('public')->delete($jobApplication->resume_path);
            $validated['resume_path'] = null;
        }

        // Handle new file upload
        if ($request->hasFile('resume')) {
            // Delete old resume if exists
            if ($jobApplication->resume_path) {
                Storage::disk('public')->delete($jobApplication->resume_path);
            }
            $validated['resume_path'] = $request->file('resume')->store('resumes', 'public');
        }

        // Remove file upload field from validated data if no new file
        if (!$request->hasFile('resume') && !$request->has('remove_resume')) {
            unset($validated['resume']);
        }
        unset($validated['remove_resume']);

        $jobApplication->update($validated);

        return redirect()->route('admin.job-applications.index')
            ->with('success', 'Lamaran pekerjaan berhasil diperbarui!');
    }

    public function destroy(JobApplication $jobApplication)
    {
        // Delete associated resume file
        if ($jobApplication->resume_path) {
            Storage::disk('public')->delete($jobApplication->resume_path);
        }

        $jobApplication->delete();

        return redirect()->route('admin.job-applications.index')
            ->with('success', 'Lamaran pekerjaan berhasil dihapus!');
    }

    public function updateStatus(Request $request, JobApplication $jobApplication)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,reviewed,interview,accepted,rejected',
            'notes' => 'nullable|string',
        ]);

        $jobApplication->update($validated);

        return redirect()->back()->with('success', 'Status lamaran berhasil diperbarui!');
    }
}
