<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MenuItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MenuController extends Controller
{
    /**
     * Display a listing of the menu items.
     */
    public function index()
    {
        $menuItems = MenuItem::with('children')
            ->main()
            ->ordered()
            ->get();

        return view('admin.menus.index', compact('menuItems'));
    }

    /**
     * Update menu visibility
     */
    public function updateVisibility(Request $request, MenuItem $menu)
    {
        // Prevent hiding system menus like home
        if ($menu->is_system && !$request->is_visible) {
            return response()->json([
                'success' => false,
                'message' => 'Menu sistem tidak dapat disembunyikan'
            ], 422);
        }

        $menu->update([
            'is_visible' => $request->boolean('is_visible')
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Status menu berhasil diperbarui'
        ]);
    }

    /**
     * Update icon visibility settings
     */
    public function updateIconSettings(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|exists:menu_items,id',
            'items.*.show_icon' => 'required|boolean',
        ]);

        DB::beginTransaction();
        try {
            foreach ($request->items as $item) {
                MenuItem::where('id', $item['id'])->update([
                    'show_icon' => $item['show_icon']
                ]);
            }
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pengaturan icon berhasil diperbarui'
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat memperbarui pengaturan icon'
            ], 500);
        }
    }

    /**
     * Update menu order
     */
    public function updateOrder(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|exists:menu_items,id',
            'items.*.sort_order' => 'required|integer|min:0',
            'items.*.parent_id' => 'nullable|exists:menu_items,id',
        ]);

        DB::beginTransaction();
        try {
            foreach ($request->items as $item) {
                MenuItem::where('id', $item['id'])->update([
                    'sort_order' => $item['sort_order'],
                    'parent_id' => $item['parent_id'] ?? null,
                ]);
            }
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Urutan menu berhasil diperbarui'
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat memperbarui urutan menu'
            ], 500);
        }
    }

    /**
     * Get menu data for API
     */
    public function getMenuData()
    {
        $menuItems = MenuItem::with('children')
            ->ordered()
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'title' => $item->title,
                    'slug' => $item->slug,
                    'icon' => $item->icon,
                    'show_icon' => $item->show_icon,
                    'parent_id' => $item->parent_id,
                    'sort_order' => $item->sort_order,
                    'is_visible' => $item->is_visible,
                    'is_system' => $item->is_system,
                    'url' => $item->url,
                ];
            });

        return response()->json($menuItems);
    }

    /**
     * Update menu settings (visibility and icon settings combined)
     */
    public function updateSettings(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $items = $request->input('items', []);
            
            foreach ($items as $item) {
                $menu = MenuItem::find($item['id']);
                
                if (!$menu) {
                    continue;
                }
                
                $updateData = [];
                
                // Only update visibility if not a system menu being hidden
                if (isset($item['is_visible'])) {
                    if (!($menu->is_system && !$item['is_visible'])) {
                        $updateData['is_visible'] = $item['is_visible'];
                    }
                }
                
                // Update icon visibility
                if (isset($item['show_icon'])) {
                    $updateData['show_icon'] = $item['show_icon'];
                }
                
                if (!empty($updateData)) {
                    $menu->update($updateData);
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Pengaturan menu berhasil disimpan'
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reset menu to default
     */
    public function reset()
    {
        DB::beginTransaction();
        try {
            // Reset all non-system menus to visible and original order
            MenuItem::where('is_system', false)->update(['is_visible' => true]);
            
            // Reset order based on original seeder data
            $defaultOrder = [
                'home' => 1,
                'about' => 2,
                'services' => 3,
                'products' => 4,
                'projects' => 5,
                'team' => 6,
                'blog' => 7,
                'contact' => 8,
            ];

            foreach ($defaultOrder as $slug => $order) {
                MenuItem::where('slug', $slug)->update(['sort_order' => $order]);
            }

            DB::commit();

            return redirect()->route('admin.menus.index')->with('success', 'Menu berhasil direset ke pengaturan default');
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('admin.menus.index')->with('error', 'Terjadi kesalahan saat mereset menu');
        }
    }
}
