<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Product::with('category')->latest();

        // Filter by category
        if ($request->category_id) {
            $query->where('product_category_id', $request->category_id);
        }

        // Filter by status
        if ($request->status !== null) {
            $query->where('is_active', $request->status);
        }

        // Search
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('sku', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
            });
        }

        $products = $query->paginate(15);
        $categories = ProductCategory::active()->orderBy('name')->get();

        return view('admin.products.index', compact('products', 'categories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = ProductCategory::active()->orderBy('name')->get();
        return view('admin.products.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'product_category_id' => 'required|exists:product_categories,id',
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:products',
            'description' => 'required|string',
            'short_description' => 'nullable|string|max:500',
            'sku' => 'nullable|string|max:255|unique:products',
            'price' => 'required|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0',
            'discount_type' => 'nullable|in:fixed,percentage',
            'stock' => 'required|integer|min:0',
            'min_order' => 'nullable|integer|min:1',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|array',
            'dimensions.length' => 'nullable|numeric|min:0',
            'dimensions.width' => 'nullable|numeric|min:0',
            'dimensions.height' => 'nullable|numeric|min:0',
            'featured_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'gallery' => 'nullable|array',
            'gallery.*' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
            'specifications' => 'nullable|array',
            'brand' => 'nullable|string|max:255',
            'condition' => 'required|in:new,used,refurbished',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
            'is_digital' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|array',
            'shopee_url' => 'nullable|url|max:500',
            'tokopedia_url' => 'nullable|url|max:500',
            'lazada_url' => 'nullable|url|max:500',
            'bukalapak_url' => 'nullable|url|max:500',
            'blibli_url' => 'nullable|url|max:500',
            'available_from' => 'nullable|date',
            'available_until' => 'nullable|date|after_or_equal:available_from',
            'tags' => 'nullable|array',
        ]);

        // Auto-generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Auto-generate SKU if not provided
        if (empty($validated['sku'])) {
            $category = ProductCategory::find($validated['product_category_id']);
            $validated['sku'] = Product::generateSku($validated['name'], $category->name);
        }

        // Handle featured image
        if ($request->hasFile('featured_image')) {
            $validated['featured_image'] = $request->file('featured_image')->store('products', 'public');
        }

        // Handle gallery images
        if ($request->hasFile('gallery')) {
            $galleryPaths = [];
            foreach ($request->file('gallery') as $file) {
                $galleryPaths[] = $file->store('products/gallery', 'public');
            }
            $validated['gallery'] = $galleryPaths;
        }

        // Handle dimensions
        if ($request->dimensions) {
            $dimensions = array_filter([
                'length' => $request->dimensions['length'] ?? null,
                'width' => $request->dimensions['width'] ?? null,
                'height' => $request->dimensions['height'] ?? null,
            ]);
            $validated['dimensions'] = !empty($dimensions) ? $dimensions : null;
        }

        // Handle specifications
        if ($request->specifications) {
            $specifications = array_filter($request->specifications, function ($spec) {
                return !empty($spec['key']) && !empty($spec['value']);
            });
            $validated['specifications'] = !empty($specifications) ? $specifications : null;
        }

        // Handle boolean fields
        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_active'] = $request->has('is_active');
        $validated['is_digital'] = $request->has('is_digital');

        // Handle tags and meta keywords
        if ($request->tags) {
            $validated['tags'] = array_filter(array_map('trim', $request->tags));
        }
        if ($request->meta_keywords) {
            $validated['meta_keywords'] = array_filter(array_map('trim', $request->meta_keywords));
        }

        Product::create($validated);

        return redirect()->route('admin.products.index')->with('success', 'Produk berhasil ditambahkan!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        $product->load('category');
        return view('admin.products.show', compact('product'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        $categories = ProductCategory::active()->orderBy('name')->get();
        return view('admin.products.edit', compact('product', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        $validated = $request->validate([
            'product_category_id' => 'required|exists:product_categories,id',
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:products,slug,' . $product->id,
            'description' => 'required|string',
            'short_description' => 'nullable|string|max:500',
            'sku' => 'nullable|string|max:255|unique:products,sku,' . $product->id,
            'price' => 'required|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0',
            'discount_type' => 'nullable|in:fixed,percentage',
            'stock' => 'required|integer|min:0',
            'min_order' => 'nullable|integer|min:1',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|array',
            'dimensions.length' => 'nullable|numeric|min:0',
            'dimensions.width' => 'nullable|numeric|min:0',
            'dimensions.height' => 'nullable|numeric|min:0',
            'featured_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'remove_featured_image' => 'nullable|boolean',
            'gallery' => 'nullable|array',
            'gallery.*' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
            'remove_gallery_images' => 'nullable|array',
            'remove_gallery_images.*' => 'integer',
            'specifications' => 'nullable|array',
            'brand' => 'nullable|string|max:255',
            'condition' => 'required|in:new,used,refurbished',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
            'is_digital' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|array',
            'shopee_url' => 'nullable|url|max:500',
            'tokopedia_url' => 'nullable|url|max:500',
            'lazada_url' => 'nullable|url|max:500',
            'bukalapak_url' => 'nullable|url|max:500',
            'blibli_url' => 'nullable|url|max:500',
            'available_from' => 'nullable|date',
            'available_until' => 'nullable|date|after_or_equal:available_from',
            'tags' => 'nullable|array',
        ]);

        // Auto-generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Handle featured image
        if ($request->hasFile('featured_image')) {
            // Delete old image
            if ($product->featured_image) {
                Storage::disk('public')->delete($product->featured_image);
            }
            $validated['featured_image'] = $request->file('featured_image')->store('products', 'public');
        } elseif ($request->has('remove_featured_image') && $request->remove_featured_image) {
            // Remove featured image if requested
            if ($product->featured_image) {
                Storage::disk('public')->delete($product->featured_image);
            }
            $validated['featured_image'] = null;
        }

        // Handle gallery images
        if ($request->hasFile('gallery')) {
            // Handle removal of individual gallery images first
            $currentGallery = $product->gallery ?? [];
            if ($request->has('remove_gallery_images') && is_array($request->remove_gallery_images)) {
                foreach ($request->remove_gallery_images as $index) {
                    if (isset($currentGallery[$index])) {
                        Storage::disk('public')->delete($currentGallery[$index]);
                        unset($currentGallery[$index]);
                    }
                }
                $currentGallery = array_values($currentGallery); // Re-index array
            }

            // Add new gallery images
            $galleryPaths = $currentGallery;
            foreach ($request->file('gallery') as $file) {
                $galleryPaths[] = $file->store('products/gallery', 'public');
            }
            $validated['gallery'] = $galleryPaths;
        } elseif ($request->has('remove_gallery_images') && is_array($request->remove_gallery_images)) {
            // Only remove specific gallery images without adding new ones
            $currentGallery = $product->gallery ?? [];
            foreach ($request->remove_gallery_images as $index) {
                if (isset($currentGallery[$index])) {
                    Storage::disk('public')->delete($currentGallery[$index]);
                    unset($currentGallery[$index]);
                }
            }
            $validated['gallery'] = array_values($currentGallery); // Re-index array
        }

        // Handle dimensions
        if ($request->dimensions) {
            $dimensions = array_filter([
                'length' => $request->dimensions['length'] ?? null,
                'width' => $request->dimensions['width'] ?? null,
                'height' => $request->dimensions['height'] ?? null,
            ]);
            $validated['dimensions'] = !empty($dimensions) ? $dimensions : null;
        }

        // Handle specifications
        if ($request->specifications) {
            $specifications = array_filter($request->specifications, function ($spec) {
                return !empty($spec['key']) && !empty($spec['value']);
            });
            $validated['specifications'] = !empty($specifications) ? $specifications : null;
        }

        // Handle boolean fields
        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_active'] = $request->has('is_active');
        $validated['is_digital'] = $request->has('is_digital');

        // Handle tags and meta keywords
        if ($request->tags) {
            $validated['tags'] = array_filter(array_map('trim', $request->tags));
        }
        if ($request->meta_keywords) {
            $validated['meta_keywords'] = array_filter(array_map('trim', $request->meta_keywords));
        }

        $product->update($validated);

        return redirect()->route('admin.products.index')->with('success', 'Produk berhasil diperbarui!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        // Delete featured image
        if ($product->featured_image) {
            Storage::disk('public')->delete($product->featured_image);
        }

        // Delete gallery images
        if ($product->gallery && is_array($product->gallery)) {
            foreach ($product->gallery as $image) {
                Storage::disk('public')->delete($image);
            }
        }

        $product->delete();

        return redirect()->route('admin.products.index')->with('success', 'Produk berhasil dihapus!');
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $validated = $request->validate([
            'action' => 'required|in:activate,deactivate,delete',
            'products' => 'required|array|min:1',
            'products.*' => 'exists:products,id',
        ]);

        $products = Product::whereIn('id', $validated['products']);

        switch ($validated['action']) {
            case 'activate':
                $products->update(['is_active' => true]);
                return redirect()->back()->with('success', 'Produk yang dipilih berhasil diaktifkan!');

            case 'deactivate':
                $products->update(['is_active' => false]);
                return redirect()->back()->with('success', 'Produk yang dipilih berhasil dinonaktifkan!');

            case 'delete':
                foreach ($products->get() as $product) {
                    // Delete images
                    if ($product->featured_image) {
                        Storage::disk('public')->delete($product->featured_image);
                    }
                    if ($product->gallery && is_array($product->gallery)) {
                        foreach ($product->gallery as $image) {
                            Storage::disk('public')->delete($image);
                        }
                    }
                }
                $products->delete();
                return redirect()->back()->with('success', 'Produk yang dipilih berhasil dihapus!');
        }

        return redirect()->back();
    }
}
