<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductCategory;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /**
     * Display a listing of products
     */
    public function index(Request $request)
    {
        $query = Product::active()->available()->with('category');
        
        // Filter by category
        if ($request->category) {
            $query->whereHas('category', function($q) use ($request) {
                $q->where('slug', $request->category);
            });
        }

        // Search
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%')
                  ->orWhere('brand', 'like', '%' . $request->search . '%');
            });
        }

        // Filter by price range
        if ($request->min_price) {
            $query->where('price', '>=', $request->min_price);
        }
        if ($request->max_price) {
            $query->where('price', '<=', $request->max_price);
        }

        // Filter by brand
        if ($request->brand) {
            $query->where('brand', $request->brand);
        }

        // Filter by condition
        if ($request->condition) {
            $query->where('condition', $request->condition);
        }

        // Sort
        $sortBy = $request->get('sort', 'latest');
        switch ($sortBy) {
            case 'name':
                $query->orderBy('name');
                break;
            case 'price_low':
                $query->orderBy('price');
                break;
            case 'price_high':
                $query->orderByDesc('price');
                break;
            case 'popular':
                $query->orderByDesc('views');
                break;
            case 'featured':
                $query->orderByDesc('is_featured')->orderBy('sort_order');
                break;
            default:
                $query->latest();
        }

        $products = $query->paginate(12)->withQueryString();
        $categories = ProductCategory::active()->withCount('activeProducts')->orderBy('sort_order')->get();

        // Get filter data
        $brands = Product::active()->whereNotNull('brand')->distinct()->pluck('brand')->sort();
        $priceRange = Product::active()->selectRaw('MIN(price) as min_price, MAX(price) as max_price')->first();

        return view('frontend.products.index', compact(
            'products', 
            'categories', 
            'brands', 
            'priceRange'
        ));
    }

    /**
     * Display the specified product
     */
    public function show($slug)
    {
        $product = Product::where('slug', $slug)
            ->active()
            ->available()
            ->with('category')
            ->firstOrFail();

        // Increment views
        $product->incrementViews();

        // Get related products
        $relatedProducts = Product::active()
            ->available()
            ->where('product_category_id', $product->product_category_id)
            ->where('id', '!=', $product->id)
            ->inStock()
            ->orderBy('sort_order')
            ->take(4)
            ->get();

        return view('frontend.products.show', compact('product', 'relatedProducts'));
    }

    /**
     * Display products by category
     */
    public function category($slug, Request $request)
    {
        $category = ProductCategory::where('slug', $slug)->active()->firstOrFail();
        
        $query = Product::active()
            ->available()
            ->where('product_category_id', $category->id)
            ->with('category');

        // Sort
        $sortBy = $request->get('sort', 'name_asc');
        switch ($sortBy) {
            case 'name_asc':
                $query->orderBy('name');
                break;
            case 'name_desc':
                $query->orderByDesc('name');
                break;
            case 'price_asc':
                $query->orderBy('price');
                break;
            case 'price_desc':
                $query->orderByDesc('price');
                break;
            case 'newest':
                $query->latest();
                break;
            case 'oldest':
                $query->oldest();
                break;
            default:
                $query->orderBy('sort_order')->orderBy('name');
        }
        
        $products = $query->paginate(12)->withQueryString();

        // Get other categories for cross-selling
        $otherCategories = ProductCategory::active()
            ->where('id', '!=', $category->id)
            ->withCount(['products' => function($q) {
                $q->active()->available();
            }])
            ->orderBy('sort_order')
            ->take(8)
            ->get();

        return view('frontend.products.category', compact('category', 'products', 'otherCategories'));
    }

    /**
     * Search products
     */
    public function search(Request $request)
    {
        $query = $request->get('q');
        $categorySlug = $request->get('category');
        $sortBy = $request->get('sort', 'relevance');
        
        // Get all categories for filter
        $categories = ProductCategory::active()
            ->withCount(['products' => function($q) {
                $q->active()->available();
            }])
            ->orderBy('sort_order')
            ->get();

        // Find selected category
        $selectedCategory = null;
        if ($categorySlug) {
            $selectedCategory = $categories->where('slug', $categorySlug)->first();
        }

        // If no search query and no category, show search page with suggestions
        if (empty($query) && empty($categorySlug)) {
            $products = collect()->paginate(0);
            $popularProducts = Product::active()
                ->available()
                ->inStock()
                ->orderByDesc('views')
                ->take(8)
                ->get();
                
            return view('frontend.products.search', compact(
                'products', 
                'query', 
                'categories', 
                'selectedCategory',
                'popularProducts'
            ));
        }

        // Build search query
        $productQuery = Product::active()->available()->with('category');

        // Filter by search term
        if (!empty($query)) {
            $productQuery->where(function ($q) use ($query) {
                $q->where('name', 'like', '%' . $query . '%')
                  ->orWhere('description', 'like', '%' . $query . '%')
                  ->orWhere('short_description', 'like', '%' . $query . '%')
                  ->orWhere('brand', 'like', '%' . $query . '%')
                  ->orWhere('sku', 'like', '%' . $query . '%')
                  ->orWhere('specifications', 'like', '%' . $query . '%');
            });
        }

        // Filter by category
        if ($selectedCategory) {
            $productQuery->where('product_category_id', $selectedCategory->id);
        }

        // Sort results
        switch ($sortBy) {
            case 'name_asc':
                $productQuery->orderBy('name');
                break;
            case 'name_desc':
                $productQuery->orderByDesc('name');
                break;
            case 'price_asc':
                $productQuery->orderBy('price');
                break;
            case 'price_desc':
                $productQuery->orderByDesc('price');
                break;
            case 'newest':
                $productQuery->latest();
                break;
            case 'relevance':
            default:
                // For relevance, prioritize exact matches in name, then featured products
                if (!empty($query)) {
                    $productQuery->orderByRaw("CASE WHEN name LIKE ? THEN 1 ELSE 2 END", ['%' . $query . '%'])
                                ->orderByDesc('is_featured')
                                ->orderBy('sort_order')
                                ->orderBy('name');
                } else {
                    $productQuery->orderByDesc('is_featured')
                                ->orderBy('sort_order')
                                ->orderBy('name');
                }
                break;
        }

        $products = $productQuery->paginate(12)->withQueryString();
        $popularProducts = collect(); // Not needed when we have search results

        return view('frontend.products.search', compact(
            'products', 
            'query', 
            'categories', 
            'selectedCategory',
            'popularProducts'
        ));
    }
}
