<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class MenuItem extends Model
{
    protected $fillable = [
        'title',
        'slug',
        'route',
        'url',
        'icon',
        'show_icon',
        'description',
        'parent_id',
        'sort_order',
        'is_visible',
        'is_system',
        'meta',
    ];

    protected $casts = [
        'is_visible' => 'boolean',
        'is_system' => 'boolean',
        'show_icon' => 'boolean',
        'meta' => 'array',
    ];

    /**
     * Get the parent menu item
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(MenuItem::class, 'parent_id');
    }

    /**
     * Get the child menu items
     */
    public function children(): HasMany
    {
        return $this->hasMany(MenuItem::class, 'parent_id')->orderBy('sort_order');
    }

    /**
     * Get visible children
     */
    public function visibleChildren(): HasMany
    {
        return $this->children()->where('is_visible', true);
    }

    /**
     * Scope for visible menu items
     */
    public function scopeVisible($query)
    {
        return $query->where('is_visible', true);
    }

    /**
     * Scope for main menu items (no parent)
     */
    public function scopeMain($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Scope for ordered menu items
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order');
    }

    /**
     * Get the URL for this menu item
     */
    public function getUrlAttribute()
    {
        if ($this->attributes['url']) {
            return $this->attributes['url'];
        }

        if ($this->route) {
            try {
                return route($this->route);
            } catch (\Exception $e) {
                return '#';
            }
        }

        return '#';
    }

    /**
     * Check if this menu item is active
     */
    public function isActive()
    {
        if ($this->route) {
            // Check if current route matches with wildcard
            $routeActive = request()->routeIs($this->route . '*');
            
            // Also check URL patterns for specific routes
            $urlActive = false;
            
            // Map routes to URL patterns for comprehensive checking
            $routeUrlMapping = [
                'home' => '',
                'about' => 'tentang-kami*',
                'services.index' => 'layanan*',
                'products.index' => 'produk*',
                'projects.index' => 'portofolio*',
                'articles.index' => 'artikel*',
                'careers.index' => 'karier*',
                'contact' => 'kontak*',
            ];
            
            // Check if current URL matches the pattern
            if (isset($routeUrlMapping[$this->route])) {
                $pattern = $routeUrlMapping[$this->route];
                if (!empty($pattern)) {
                    $urlActive = request()->is($pattern);
                }
            }
            
            return $routeActive || $urlActive;
        }

        return false;
    }

    /**
     * Check if this menu item has active children
     */
    public function hasActiveChildren()
    {
        return $this->visibleChildren->contains(function ($child) {
            return $child->isActive() || $child->hasActiveChildren();
        });
    }
}
