<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class Product extends Model
{
    protected $fillable = [
        'product_category_id',
        'name',
        'slug',
        'description',
        'short_description',
        'sku',
        'price',
        'discount_price',
        'discount_type',
        'stock',
        'min_order',
        'weight',
        'dimensions',
        'featured_image',
        'gallery',
        'specifications',
        'brand',
        'condition',
        'is_featured',
        'is_active',
        'is_digital',
        'sort_order',
        'views',
        'sales_count',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'shopee_url',
        'tokopedia_url',
        'lazada_url',
        'bukalapak_url',
        'blibli_url',
        'available_from',
        'available_until',
        'tags',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'discount_price' => 'decimal:2',
        'weight' => 'decimal:2',
        'dimensions' => 'array',
        'gallery' => 'array',
        'specifications' => 'array',
        'meta_keywords' => 'array',
        'tags' => 'array',
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
        'is_digital' => 'boolean',
        'views' => 'integer',
        'sales_count' => 'integer',
        'sort_order' => 'integer',
        'stock' => 'integer',
        'min_order' => 'integer',
        'available_from' => 'datetime',
        'available_until' => 'datetime',
    ];

    /**
     * Get the category that owns the product
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(ProductCategory::class, 'product_category_id');
    }

    /**
     * Scope for active products
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for featured products
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope for in stock products
     */
    public function scopeInStock($query)
    {
        return $query->where('stock', '>', 0);
    }

    /**
     * Scope for available products (within date range)
     */
    public function scopeAvailable($query)
    {
        $now = now();
        return $query->where(function ($q) use ($now) {
            $q->where('available_from', '<=', $now)
              ->orWhereNull('available_from');
        })->where(function ($q) use ($now) {
            $q->where('available_until', '>=', $now)
              ->orWhereNull('available_until');
        });
    }

    /**
     * Get the route key for the model
     */
    public function getRouteKeyName(): string
    {
        return 'slug';
    }

    /**
     * Get featured image URL
     */
    public function getFeaturedImageUrlAttribute(): ?string
    {
        return $this->featured_image ? asset('storage/' . $this->featured_image) : null;
    }

    /**
     * Get gallery image URLs
     */
    public function getGalleryUrlsAttribute(): array
    {
        if (!$this->gallery || !is_array($this->gallery)) {
            return [];
        }

        return array_map(function ($image) {
            return asset('storage/' . $image);
        }, $this->gallery);
    }

    /**
     * Get all images (featured + gallery)
     */
    public function getAllImagesAttribute(): array
    {
        $images = [];
        
        if ($this->featured_image) {
            $images[] = $this->featured_image_url;
        }
        
        return array_merge($images, $this->gallery_urls);
    }

    /**
     * Get final price (with discount if applicable)
     */
    public function getFinalPriceAttribute(): float
    {
        if (!$this->discount_price) {
            return $this->price;
        }

        if ($this->discount_type === 'percentage') {
            return $this->price - ($this->price * $this->discount_price / 100);
        }

        return $this->price - $this->discount_price;
    }

    /**
     * Get discount amount
     */
    public function getDiscountAmountAttribute(): float
    {
        return $this->price - $this->final_price;
    }

    /**
     * Get discount percentage
     */
    public function getDiscountPercentageAttribute(): float
    {
        if ($this->price <= 0) return 0;
        return ($this->discount_amount / $this->price) * 100;
    }

    /**
     * Check if product has discount
     */
    public function getHasDiscountAttribute(): bool
    {
        return $this->discount_price > 0;
    }

    /**
     * Check if product is in stock
     */
    public function getIsInStockAttribute(): bool
    {
        return $this->stock > 0;
    }

    /**
     * Get stock status
     */
    public function getStockStatusAttribute(): string
    {
        if ($this->is_digital) {
            return 'Digital Product';
        }

        if ($this->stock <= 0) {
            return 'Out of Stock';
        }

        if ($this->stock <= 10) {
            return 'Low Stock';
        }

        return 'In Stock';
    }

    /**
     * Get formatted price
     */
    public function getFormattedPriceAttribute(): string
    {
        return 'Rp ' . number_format($this->price, 0, ',', '.');
    }

    /**
     * Get formatted final price
     */
    public function getFormattedFinalPriceAttribute(): string
    {
        return 'Rp ' . number_format($this->final_price, 0, ',', '.');
    }

    /**
     * Accessor for marketplace links
     */
    public function getMarketplaceLinksAttribute()
    {
        $links = [];
        
        if ($this->shopee_url) {
            $links['shopee'] = $this->shopee_url;
        }
        
        if ($this->tokopedia_url) {
            $links['tokopedia'] = $this->tokopedia_url;
        }
        
        if ($this->lazada_url) {
            $links['lazada'] = $this->lazada_url;
        }
        
        if ($this->bukalapak_url) {
            $links['bukalapak'] = $this->bukalapak_url;
        }
        
        if ($this->blibli_url) {
            $links['blibli'] = $this->blibli_url;
        }
        
        return !empty($links) ? $links : null;
    }

    /**
     * Increment views
     */
    public function incrementViews(): void
    {
        $this->increment('views');
    }

    /**
     * Increment sales count
     */
    public function incrementSales(int $quantity = 1): void
    {
        $this->increment('sales_count', $quantity);
    }

    /**
     * Generate unique SKU
     */
    public static function generateSku(string $name, ?string $category = null): string
    {
        $prefix = $category ? Str::upper(Str::substr($category, 0, 3)) : 'PRD';
        $slug = Str::upper(Str::substr(Str::slug($name, ''), 0, 6));
        $timestamp = now()->format('ymd');
        
        $baseSku = $prefix . '-' . $slug . '-' . $timestamp;
        $sku = $baseSku;
        $counter = 1;

        while (self::where('sku', $sku)->exists()) {
            $sku = $baseSku . '-' . str_pad($counter, 2, '0', STR_PAD_LEFT);
            $counter++;
        }

        return $sku;
    }

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($product) {
            // Auto-generate slug if not provided
            if (empty($product->slug)) {
                $product->slug = Str::slug($product->name);
            }

            // Auto-generate SKU if not provided
            if (empty($product->sku)) {
                $product->sku = self::generateSku($product->name);
            }
        });

        static::updating(function ($product) {
            // Update slug if name changed
            if ($product->isDirty('name') && empty($product->getOriginal('slug'))) {
                $product->slug = Str::slug($product->name);
            }
        });
    }
}
