<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class WebsiteVisitor extends Model
{
    use HasFactory;

    protected $fillable = [
        'ip_address',
        'user_agent',
        'url',
        'referer',
        'country',
        'city',
        'device_type',
        'browser',
        'os',
        'visit_date',
    ];

    protected $casts = [
        'visit_date' => 'date',
    ];

    /**
     * Check if visitor already visited today
     */
    public static function hasVisitedToday($ipAddress)
    {
        return self::where('ip_address', $ipAddress)
                   ->where('visit_date', Carbon::today())
                   ->exists();
    }

    /**
     * Record visitor
     */
    public static function recordVisit($request)
    {
        $ipAddress = $request->ip();
        
        // Cek apakah IP sudah mengunjungi hari ini
        if (self::hasVisitedToday($ipAddress)) {
            return false; // Tidak record jika sudah mengunjungi hari ini
        }

        // Parse user agent untuk mendapatkan info device
        $userAgent = $request->userAgent();
        $deviceInfo = self::parseUserAgent($userAgent);

        return self::create([
            'ip_address' => $ipAddress,
            'user_agent' => $userAgent,
            'url' => $request->fullUrl(),
            'referer' => $request->header('referer'),
            'country' => null, // Bisa diintegrasikan dengan GeoIP
            'city' => null,
            'device_type' => $deviceInfo['device_type'],
            'browser' => $deviceInfo['browser'],
            'os' => $deviceInfo['os'],
            'visit_date' => Carbon::today(),
        ]);
    }

    /**
     * Parse user agent untuk mendapatkan info device
     */
    public static function parseUserAgent($userAgent)
    {
        $deviceType = 'desktop';
        $browser = 'Unknown';
        $os = 'Unknown';

        if (preg_match('/Mobile|Android|iPhone|iPad/', $userAgent)) {
            if (preg_match('/iPad/', $userAgent)) {
                $deviceType = 'tablet';
            } else {
                $deviceType = 'mobile';
            }
        }

        // Detect browser
        if (preg_match('/Chrome/', $userAgent)) {
            $browser = 'Chrome';
        } elseif (preg_match('/Firefox/', $userAgent)) {
            $browser = 'Firefox';
        } elseif (preg_match('/Safari/', $userAgent)) {
            $browser = 'Safari';
        } elseif (preg_match('/Edge/', $userAgent)) {
            $browser = 'Edge';
        }

        // Detect OS
        if (preg_match('/Windows/', $userAgent)) {
            $os = 'Windows';
        } elseif (preg_match('/Macintosh/', $userAgent)) {
            $os = 'MacOS';
        } elseif (preg_match('/Linux/', $userAgent)) {
            $os = 'Linux';
        } elseif (preg_match('/Android/', $userAgent)) {
            $os = 'Android';
        } elseif (preg_match('/iPhone|iPad/', $userAgent)) {
            $os = 'iOS';
        }

        return [
            'device_type' => $deviceType,
            'browser' => $browser,
            'os' => $os,
        ];
    }

    /**
     * Get visitors count by date range
     */
    public static function getVisitorsByDateRange($startDate, $endDate)
    {
        return self::whereBetween('visit_date', [$startDate, $endDate])
                   ->selectRaw('visit_date, COUNT(*) as visitors')
                   ->groupBy('visit_date')
                   ->orderBy('visit_date')
                   ->get();
    }

    /**
     * Get total visitors
     */
    public static function getTotalVisitors()
    {
        return self::count();
    }

    /**
     * Get today's visitors
     */
    public static function getTodayVisitors()
    {
        return self::where('visit_date', Carbon::today())->count();
    }

    /**
     * Get this month visitors
     */
    public static function getThisMonthVisitors()
    {
        return self::whereMonth('visit_date', Carbon::now()->month)
                   ->whereYear('visit_date', Carbon::now()->year)
                   ->count();
    }
}
