@extends('layouts.admin')

@section('title', 'Kelola Menu')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <!-- Header -->
    <div class="mb-8">
        <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between">
            <div>
                <h1 class="text-3xl font-bold text-gray-900">Kelola Menu Website</h1>
                <p class="mt-1 text-sm text-gray-600">Atur tampilan dan urutan menu navigasi website</p>
            </div>
            <div class="mt-4 sm:mt-0 flex space-x-3">
                <button onclick="openIconModal()" 
                        class="inline-flex items-center px-4 py-2 border border-purple-300 rounded-lg text-sm font-medium text-purple-700 bg-white hover:bg-purple-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-purple-500 transition duration-150 ease-in-out">
                    <i class="fas fa-cog mr-2"></i>
                    Pengaturan Menu
                </button>
                <button onclick="resetMenu()" 
                        class="inline-flex items-center px-4 py-2 border border-gray-300 rounded-lg text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 transition duration-150 ease-in-out">
                    <i class="fas fa-undo mr-2"></i>
                    Reset Default
                </button>
                <button onclick="saveMenuOrder()" 
                        class="inline-flex items-center px-4 py-2 bg-blue-600 border border-transparent rounded-lg text-sm font-medium text-white hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 transition duration-150 ease-in-out">
                    <i class="fas fa-save mr-2"></i>
                    Simpan Urutan
                </button>
            </div>
        </div>
    </div>

    <!-- Instructions -->
    <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
        <div class="flex">
            <div class="flex-shrink-0">
                <i class="fas fa-info-circle text-blue-400"></i>
            </div>
            <div class="ml-3">
                <p class="text-sm text-blue-700">
                    <strong>Cara Penggunaan:</strong>
                </p>
                <ul class="mt-2 text-sm text-blue-600 list-disc list-inside space-y-1">
                    <li>Seret dan lepas menu untuk mengubah urutan</li>
                    <li>Seret menu ke dalam menu lain untuk membuat submenu</li>
                    <li>Gunakan toggle untuk menampilkan/menyembunyikan menu</li>
                    <li>Klik "Pengaturan Menu" untuk mengatur visibility menu dan tampilan icon</li>
                    <li>Menu "Beranda" tidak dapat disembunyikan karena merupakan menu sistem</li>
                </ul>
            </div>
        </div>
    </div>

    <!-- Menu Management -->
    <div class="bg-white shadow-sm border border-gray-200 rounded-lg">
        <div class="px-6 py-4 border-b border-gray-200">
            <h3 class="text-lg leading-6 font-medium text-gray-900">
                <i class="fas fa-bars mr-2"></i>
                Struktur Menu
            </h3>
        </div>
        <div class="p-6">
            <div id="menu-container">
                <ul id="menu-list" class="space-y-2">
                    @foreach($menuItems as $menu)
                        <li class="menu-item" data-id="{{ $menu->id }}" data-parent-id="{{ $menu->parent_id }}">
                            <div class="flex items-center justify-between p-4 bg-gray-50 border border-gray-200 rounded-lg hover:bg-gray-100 transition-colors cursor-move">
                                <div class="flex items-center space-x-4">
                                    <div class="drag-handle text-gray-400 hover:text-gray-600">
                                        <i class="fas fa-grip-vertical"></i>
                                    </div>
                                    <div class="flex items-center space-x-3">
                                        @if($menu->icon)
                                            <i class="{{ $menu->icon }} text-gray-600"></i>
                                        @endif
                                        <div>
                                            <h4 class="text-sm font-medium text-gray-900">{{ $menu->title }}</h4>
                                            <p class="text-xs text-gray-500">{{ $menu->description }}</p>
                                        </div>
                                    </div>
                                    @if($menu->is_system)
                                        <span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium bg-blue-100 text-blue-800">
                                            <i class="fas fa-shield-alt mr-1"></i>
                                            Sistem
                                        </span>
                                    @endif
                                </div>
                                <div class="flex items-center space-x-4">
                                    <div class="flex items-center">
                                        <label class="relative inline-flex items-center cursor-pointer">
                                            <input type="checkbox" 
                                                   class="sr-only peer visibility-toggle" 
                                                   data-menu-id="{{ $menu->id }}"
                                                   {{ $menu->is_visible ? 'checked' : '' }}
                                                   {{ $menu->is_system ? 'disabled' : '' }}>
                                            <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600 {{ $menu->is_system ? 'opacity-50 cursor-not-allowed' : '' }}"></div>
                                        </label>
                                        <span class="ml-3 text-sm text-gray-900">
                                            {{ $menu->is_visible ? 'Tampil' : 'Tersembunyi' }}
                                        </span>
                                    </div>
                                </div>
                            </div>
                            
                            @if($menu->children->count() > 0)
                                <ul class="ml-8 mt-2 space-y-2">
                                    @foreach($menu->children as $child)
                                        <li class="menu-item" data-id="{{ $child->id }}" data-parent-id="{{ $child->parent_id }}">
                                            <div class="flex items-center justify-between p-3 bg-gray-25 border border-gray-150 rounded-lg hover:bg-gray-50 transition-colors cursor-move">
                                                <div class="flex items-center space-x-4">
                                                    <div class="drag-handle text-gray-400 hover:text-gray-600">
                                                        <i class="fas fa-grip-vertical text-sm"></i>
                                                    </div>
                                                    <div class="flex items-center space-x-3">
                                                        @if($child->icon)
                                                            <i class="{{ $child->icon }} text-gray-500 text-sm"></i>
                                                        @endif
                                                        <div>
                                                            <h5 class="text-sm font-medium text-gray-800">{{ $child->title }}</h5>
                                                            <p class="text-xs text-gray-500">{{ $child->description }}</p>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="flex items-center space-x-4">
                                                    <div class="flex items-center">
                                                        <label class="relative inline-flex items-center cursor-pointer">
                                                            <input type="checkbox" 
                                                                   class="sr-only peer visibility-toggle" 
                                                                   data-menu-id="{{ $child->id }}"
                                                                   {{ $child->is_visible ? 'checked' : '' }}>
                                                            <div class="w-9 h-5 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[1px] after:left-[1px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-4 after:w-4 after:transition-all peer-checked:bg-blue-600"></div>
                                                        </label>
                                                        <span class="ml-2 text-xs text-gray-800">
                                                            {{ $child->is_visible ? 'Tampil' : 'Tersembunyi' }}
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                        </li>
                                    @endforeach
                                </ul>
                            @endif
                        </li>
                    @endforeach
                </ul>
            </div>
            
            <div class="mt-6 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-triangle text-yellow-400"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm text-yellow-700">
                            <strong>Perhatian:</strong> Perubahan urutan menu akan otomatis tersimpan saat Anda seret dan lepas. 
                            Perubahan visibility akan tersimpan otomatis saat toggle diubah.
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Reset Confirmation Modal -->
<div id="resetModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50 hidden">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-yellow-100">
                <i class="fas fa-exclamation-triangle text-yellow-600 text-xl"></i>
            </div>
            <div class="mt-3 text-center">
                <h3 class="text-lg leading-6 font-medium text-gray-900">Reset Menu ke Default</h3>
                <div class="mt-2 px-7 py-3">
                    <p class="text-sm text-gray-500">
                        Apakah Anda yakin ingin mereset semua pengaturan menu ke pengaturan default?
                        <br><br>
                        <span class="text-yellow-600">Semua perubahan urutan dan visibility akan dikembalikan ke pengaturan awal.</span>
                    </p>
                </div>
                <div class="flex justify-center space-x-4 px-7 py-3">
                    <button type="button" onclick="closeResetModal()"
                            class="px-4 py-2 bg-gray-300 text-gray-800 text-sm font-medium rounded-md hover:bg-gray-400 focus:outline-none focus:ring-2 focus:ring-gray-300">
                        Batal
                    </button>
                    <form id="reset-form" action="{{ route('admin.menus.reset') }}" method="POST" class="inline">
                        @csrf
                        <button type="submit"
                                class="px-4 py-2 bg-yellow-600 text-white text-sm font-medium rounded-md hover:bg-yellow-700 focus:outline-none focus:ring-2 focus:ring-yellow-500">
                            Reset
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Menu Settings Modal -->
<div id="iconModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50 hidden">
    <div class="relative top-10 mx-auto p-5 border w-3/4 max-w-5xl shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-lg leading-6 font-medium text-gray-900">
                    <i class="fas fa-cog mr-2 text-purple-600"></i>
                    Pengaturan Menu Website
                </h3>
                <button onclick="closeIconModal()" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>
            <div class="mb-4 p-4 bg-blue-50 border border-blue-200 rounded-lg">
                <div class="flex items-start space-x-3">
                    <i class="fas fa-info-circle text-blue-500 mt-0.5"></i>
                    <div>
                        <p class="text-sm text-blue-800 mb-2">
                            <strong>Cara Menggunakan:</strong>
                        </p>
                        <ul class="text-sm text-blue-700 space-y-1">
                            <li><strong>Menu Visibility:</strong> Toggle hijau untuk show/hide menu di website. Centang = tampil, tidak centang = tersembunyi.</li>
                            <li><strong>Icon Visibility:</strong> Toggle biru untuk show/hide icon menu. Gunakan tombol "Tampilkan/Sembunyikan Semua" untuk mengatur semua icon sekaligus.</li>
                            <li><strong>Contoh:</strong> Untuk menyembunyikan menu "Produk", matikan toggle hijau di baris menu Produk.</li>
                        </ul>
                    </div>
                </div>
            </div>
            
            <div class="max-h-96 overflow-y-auto border border-gray-200 rounded-lg">
                <div id="iconSettingsList" class="divide-y divide-gray-200">
                    <!-- Menu settings akan di-load di sini via JavaScript -->
                </div>
            </div>
            
            <div class="flex justify-between items-center mt-6 pt-4 border-t border-gray-200">
                <div class="flex space-x-3">
                    <div class="flex items-center space-x-2 text-sm text-gray-600">
                        <i class="fas fa-hand-pointer text-green-600"></i>
                        <span>Klik toggle hijau untuk show/hide menu secara individual</span>
                    </div>
                    <div class="border-l border-gray-300 pl-3 flex space-x-2">
                        <span class="text-sm font-medium text-gray-700">Icon (Bulk Action):</span>
                        <button onclick="toggleAllIcons(true)" 
                                class="px-3 py-1 bg-blue-600 text-white text-xs font-medium rounded hover:bg-blue-700 transition-colors">
                            <i class="fas fa-eye mr-1"></i>
                            Tampilkan Semua
                        </button>
                        <button onclick="toggleAllIcons(false)" 
                                class="px-3 py-1 bg-gray-600 text-white text-xs font-medium rounded hover:bg-gray-700 transition-colors">
                            <i class="fas fa-eye-slash mr-1"></i>
                            Sembunyikan Semua
                        </button>
                    </div>
                </div>
                <div class="flex space-x-3">
                    <button onclick="closeIconModal()" 
                            class="px-4 py-2 bg-gray-300 text-gray-800 text-sm font-medium rounded-md hover:bg-gray-400 focus:outline-none focus:ring-2 focus:ring-gray-300">
                        Batal
                    </button>
                    <button onclick="saveMenuSettings()" 
                            class="px-4 py-2 bg-purple-600 text-white text-sm font-medium rounded-md hover:bg-purple-700 focus:outline-none focus:ring-2 focus:ring-purple-500">
                        <i class="fas fa-save mr-1"></i>
                        Simpan Pengaturan
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Include SortableJS -->
<script src="https://cdn.jsdelivr.net/npm/sortablejs@latest/Sortable.min.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize sortable for main menu
    const menuList = document.getElementById('menu-list');
    const sortable = Sortable.create(menuList, {
        group: {
            name: 'menu',
            pull: true,
            put: function(to, from, dragEl) {
                // Don't allow nesting more than 2 levels
                if (to.el.closest('.menu-item')) {
                    return false;
                }
                return true;
            }
        },
        animation: 150,
        handle: '.drag-handle',
        ghostClass: 'sortable-ghost',
        chosenClass: 'sortable-chosen',
        dragClass: 'sortable-drag',
        filter: '.no-drag',
        preventOnFilter: false,
        onEnd: function(evt) {
            updateMenuOrder();
        }
    });

    // Initialize sortable for existing submenus
    initializeSubmenus();

    // Handle visibility toggle
    document.querySelectorAll('.visibility-toggle').forEach(function(toggle) {
        toggle.addEventListener('change', function() {
            const menuId = this.dataset.menuId;
            const isVisible = this.checked;
            
            updateVisibility(menuId, isVisible);
        });
    });
});

function initializeSubmenus() {
    document.querySelectorAll('#menu-list ul').forEach(function(submenu) {
        if (!submenu.sortableInstance) {
            submenu.sortableInstance = Sortable.create(submenu, {
                group: {
                    name: 'submenu',
                    pull: true,
                    put: true
                },
                animation: 150,
                handle: '.drag-handle',
                ghostClass: 'sortable-ghost',
                chosenClass: 'sortable-chosen',
                dragClass: 'sortable-drag',
                onEnd: function(evt) {
                    updateMenuOrder();
                }
            });
        }
    });
}

function updateMenuOrder() {
    const items = [];
    let sortOrder = 1;

    document.querySelectorAll('#menu-list > .menu-item').forEach(function(item) {
        const id = item.dataset.id;
        items.push({
            id: parseInt(id),
            sort_order: sortOrder++,
            parent_id: null
        });

        // Handle submenus
        let subSortOrder = 1;
        const submenuContainer = item.querySelector('ul');
        if (submenuContainer) {
            submenuContainer.querySelectorAll('.menu-item').forEach(function(subItem) {
                const subId = subItem.dataset.id;
                items.push({
                    id: parseInt(subId),
                    sort_order: subSortOrder++,
                    parent_id: parseInt(id)
                });
                // Update the data attribute
                subItem.dataset.parentId = id;
            });
        }
    });

    // Send AJAX request to update order
    fetch('{{ route("admin.menus.update-order") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({ items: items })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification(data.message, 'success');
            // Reinitialize submenus after order change
            setTimeout(() => {
                initializeSubmenus();
            }, 100);
        } else {
            showNotification(data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Terjadi kesalahan saat memperbarui urutan menu', 'error');
    });
}

function updateVisibility(menuId, isVisible) {
    fetch(`{{ url('admin/menus') }}/${menuId}/visibility`, {
        method: 'PATCH',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({ is_visible: isVisible })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification(data.message, 'success');
            
            // Update visibility text
            const toggle = document.querySelector(`[data-menu-id="${menuId}"]`);
            const visibilityText = toggle.closest('.flex').querySelector('span:last-child');
            visibilityText.textContent = isVisible ? 'Tampil' : 'Tersembunyi';
        } else {
            showNotification(data.message, 'error');
            // Revert toggle if failed
            document.querySelector(`[data-menu-id="${menuId}"]`).checked = !isVisible;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Terjadi kesalahan saat mengubah visibility menu', 'error');
        // Revert toggle if failed
        document.querySelector(`[data-menu-id="${menuId}"]`).checked = !isVisible;
    });
}

function saveMenuOrder() {
    updateMenuOrder();
}

function resetMenu() {
    document.getElementById('resetModal').classList.remove('hidden');
}

function closeResetModal() {
    document.getElementById('resetModal').classList.add('hidden');
}

// Icon Modal Functions
let menuData = [];

function openIconModal() {
    document.getElementById('iconModal').classList.remove('hidden');
    loadIconSettings();
}

function closeIconModal() {
    document.getElementById('iconModal').classList.add('hidden');
}

function loadIconSettings() {
    // Show loading state
    const container = document.getElementById('iconSettingsList');
    container.innerHTML = '<div class="p-4 text-center text-gray-500">Memuat data menu...</div>';
    
    fetch('{{ route("admin.menus.data") }}')
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(data => {
            menuData = data;
            renderMenuSettings(data);
        })
        .catch(error => {
            console.error('Error:', error);
            container.innerHTML = '<div class="p-4 text-center text-red-500">Gagal memuat data menu</div>';
            showNotification('Gagal memuat data menu', 'error');
        });
}

function renderMenuSettings(data) {
    const container = document.getElementById('iconSettingsList');
    container.innerHTML = '';
    
    data.forEach(menu => {
        if (menu.parent_id === null) { // Main menu only
            const menuItem = createMenuSettingItem(menu);
            container.appendChild(menuItem);
            
            // Add children if exists
            const children = data.filter(item => item.parent_id === menu.id);
            if (children.length > 0) {
                children.forEach(child => {
                    const childItem = createMenuSettingItem(child, true);
                    container.appendChild(childItem);
                });
            }
        }
    });
}

function createMenuSettingItem(menu, isChild = false) {
    const div = document.createElement('div');
    div.className = `p-4 ${isChild ? 'bg-gray-50 ml-8' : 'bg-white'} hover:bg-gray-100 transition-colors`;
    div.dataset.menuId = menu.id;
    
    const isSystemMenu = menu.is_system || false;
    
    div.innerHTML = `
        <div class="flex items-center justify-between">
            <div class="flex items-center space-x-3">
                ${isChild ? '<i class="fas fa-arrow-right text-gray-400 text-sm mr-2"></i>' : ''}
                ${menu.icon ? `<i class="${menu.icon} text-gray-600"></i>` : '<i class="fas fa-circle text-gray-300"></i>'}
                <div>
                    <div class="flex items-center space-x-2">
                        <h4 class="text-sm font-medium text-gray-900">${menu.title}</h4>
                        ${isSystemMenu ? '<span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium bg-blue-100 text-blue-800"><i class="fas fa-shield-alt mr-1"></i>Sistem</span>' : ''}
                    </div>
                    <p class="text-xs text-gray-500">${menu.icon || 'Tidak ada icon'}</p>
                </div>
            </div>
            <div class="flex items-center space-x-8">
                <!-- Menu Visibility Toggle -->
                <div class="flex items-center space-x-3">
                    <span class="text-sm text-gray-700 font-medium">Menu:</span>
                    <span class="text-sm font-semibold menu-status ${menu.is_visible ? 'text-green-600' : 'text-red-600'}">
                        ${menu.is_visible ? 'Tampil' : 'Tersembunyi'}
                    </span>
                    <div class="relative">
                        <input type="checkbox" 
                               id="menu-${menu.id}" 
                               class="menu-toggle" 
                               data-menu-id="${menu.id}"
                               ${menu.is_visible ? 'checked' : ''}
                               ${isSystemMenu ? 'disabled' : ''}>
                        <label for="menu-${menu.id}" class="toggle-switch ${isSystemMenu ? 'disabled' : ''}">
                            <span class="toggle-slider"></span>
                        </label>
                    </div>
                </div>
                
                <!-- Icon Visibility Toggle -->
                <div class="flex items-center space-x-3">
                    <span class="text-sm text-gray-700 font-medium">Icon:</span>
                    <span class="text-sm font-semibold icon-status ${menu.show_icon ? 'text-blue-600' : 'text-gray-600'}">
                        ${menu.show_icon ? 'Tampil' : 'Tersembunyi'}
                    </span>
                    <div class="relative">
                        <input type="checkbox" 
                               id="icon-${menu.id}" 
                               class="icon-toggle" 
                               data-menu-id="${menu.id}"
                               ${menu.show_icon ? 'checked' : ''}
                               ${!menu.icon ? 'disabled' : ''}>
                        <label for="icon-${menu.id}" class="toggle-switch-blue ${!menu.icon ? 'disabled' : ''}">
                            <span class="toggle-slider-blue"></span>
                        </label>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // Add event listeners for toggles
    const menuToggle = div.querySelector('.menu-toggle');
    if (menuToggle && !menuToggle.disabled) {
        menuToggle.addEventListener('change', function() {
            updateMenuToggleText(this);
        });
    }
    
    const iconToggle = div.querySelector('.icon-toggle');
    if (iconToggle && !iconToggle.disabled) {
        iconToggle.addEventListener('change', function() {
            updateIconToggleText(this);
        });
    }
    
    return div;
}

function updateMenuToggleText(toggle) {
    const menuItem = toggle.closest('[data-menu-id]');
    const statusText = menuItem.querySelector('.menu-status');
    if (statusText) {
        if (toggle.checked) {
            statusText.textContent = 'Tampil';
            statusText.className = 'text-sm font-semibold menu-status text-green-600';
        } else {
            statusText.textContent = 'Tersembunyi';
            statusText.className = 'text-sm font-semibold menu-status text-red-600';
        }
    }
}

function updateIconToggleText(toggle) {
    const menuItem = toggle.closest('[data-menu-id]');
    const statusText = menuItem.querySelector('.icon-status');
    if (statusText) {
        if (toggle.checked) {
            statusText.textContent = 'Tampil';
            statusText.className = 'text-sm font-semibold icon-status text-blue-600';
        } else {
            statusText.textContent = 'Tersembunyi';
            statusText.className = 'text-sm font-semibold icon-status text-gray-600';
        }
    }
}

function toggleAllIcons(show) {
    const toggles = document.querySelectorAll('.icon-toggle:not([disabled])');
    toggles.forEach(toggle => {
        toggle.checked = show;
        updateIconToggleText(toggle);
    });
}

function saveMenuSettings() {
    const menuToggles = document.querySelectorAll('.menu-toggle');
    const iconToggles = document.querySelectorAll('.icon-toggle');
    const items = [];
    
    // Collect menu visibility settings
    menuToggles.forEach(toggle => {
        const menuId = parseInt(toggle.dataset.menuId);
        const existingItem = items.find(item => item.id === menuId);
        
        if (existingItem) {
            existingItem.is_visible = toggle.checked;
        } else {
            items.push({
                id: menuId,
                is_visible: toggle.checked,
                show_icon: true // default value
            });
        }
    });
    
    // Collect icon visibility settings
    iconToggles.forEach(toggle => {
        const menuId = parseInt(toggle.dataset.menuId);
        const existingItem = items.find(item => item.id === menuId);
        
        if (existingItem) {
            existingItem.show_icon = toggle.checked;
        } else {
            items.push({
                id: menuId,
                show_icon: toggle.checked,
                is_visible: true // default value
            });
        }
    });
    
    const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
    
    fetch('{{ route("admin.menus.update-settings") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken
        },
        body: JSON.stringify({ items: items })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification(data.message, 'success');
            closeIconModal();
            // Reload page to see changes
            setTimeout(() => {
                window.location.reload();
            }, 1000);
        } else {
            showNotification(data.message || 'Terjadi kesalahan', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Terjadi kesalahan saat menyimpan pengaturan', 'error');
    });
}

function showNotification(message, type) {
    // Remove any existing notifications
    const existingNotification = document.querySelector('.notification-popup');
    if (existingNotification) {
        existingNotification.remove();
    }

    // Create simple notification element
    let bgColor = 'bg-green-500';
    if (type === 'error') bgColor = 'bg-red-500';
    if (type === 'warning') bgColor = 'bg-yellow-500';
    
    const notification = document.createElement('div');
    notification.className = `notification-popup fixed top-4 right-4 z-50 px-4 py-3 rounded-lg shadow-lg text-white text-sm font-medium transition-all duration-300 transform translate-x-full opacity-0 ${bgColor}`;
    
    // Add icon and message
    let icon = 'fas fa-check-circle';
    if (type === 'error') icon = 'fas fa-exclamation-circle';
    if (type === 'warning') icon = 'fas fa-exclamation-triangle';
    
    notification.innerHTML = `
        <div class="flex items-center">
            <i class="${icon} mr-2"></i>
            <span>${message}</span>
        </div>
    `;

    document.body.appendChild(notification);

    // Animate in
    setTimeout(() => {
        notification.classList.remove('translate-x-full', 'opacity-0');
    }, 100);

    // Auto remove after 3 seconds
    setTimeout(() => {
        notification.classList.add('translate-x-full', 'opacity-0');
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 300);
    }, 3000);
}

// Close modal when clicking outside
document.getElementById('resetModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeResetModal();
    }
});

// Close icon modal when clicking outside
document.getElementById('iconModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeIconModal();
    }
});

// Close modal on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeResetModal();
        closeIconModal();
    }
});
</script>

<style>
.sortable-ghost {
    opacity: 0.4;
    background: #e5e7eb;
}

.sortable-chosen {
    cursor: grabbing;
    background: #dbeafe;
    border-color: #3b82f6;
}

.sortable-drag {
    transform: rotate(5deg);
    box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
}

.menu-item:hover .drag-handle {
    color: #4B5563;
}

.menu-item .drag-handle {
    cursor: grab;
    transition: color 0.2s ease;
}

.menu-item .drag-handle:active {
    cursor: grabbing;
}

/* Submenu styling */
.menu-item ul {
    border-left: 2px solid #e5e7eb;
    margin-left: 1rem;
    padding-left: 1rem;
}

.menu-item ul .menu-item {
    position: relative;
}

.menu-item ul .menu-item::before {
    content: '';
    position: absolute;
    left: -1rem;
    top: 50%;
    width: 0.75rem;
    height: 2px;
    background: #e5e7eb;
}

/* Drop zone indicators */
.sortable-drop-zone {
    background: #dbeafe;
    border: 2px dashed #3b82f6;
    min-height: 3rem;
    border-radius: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #3b82f6;
    font-size: 0.875rem;
    margin: 0.25rem 0;
}

/* Custom Toggle Switches */
.toggle-switch, .toggle-switch-blue {
    position: relative;
    display: inline-block;
    width: 52px;
    height: 28px;
    background-color: #cbd5e0;
    border-radius: 14px;
    transition: background-color 0.3s;
    cursor: pointer;
    border: 2px solid #e2e8f0;
}

.toggle-switch:hover, .toggle-switch-blue:hover {
    background-color: #a0aec0;
}

.toggle-switch.disabled, .toggle-switch-blue.disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.toggle-slider, .toggle-slider-blue {
    position: absolute;
    top: 2px;
    left: 2px;
    width: 20px;
    height: 20px;
    background-color: white;
    border-radius: 50%;
    transition: transform 0.3s;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
}

/* Green Toggle for Menu Visibility */
.menu-toggle:checked + .toggle-switch {
    background-color: #48bb78;
    border-color: #38a169;
}

.menu-toggle:checked + .toggle-switch .toggle-slider {
    transform: translateX(24px);
}

.menu-toggle:focus + .toggle-switch {
    box-shadow: 0 0 0 3px rgba(72, 187, 120, 0.3);
}

/* Blue Toggle for Icon Visibility */
.icon-toggle:checked + .toggle-switch-blue {
    background-color: #4299e1;
    border-color: #3182ce;
}

.icon-toggle:checked + .toggle-switch-blue .toggle-slider-blue {
    transform: translateX(24px);
}

.icon-toggle:focus + .toggle-switch-blue {
    box-shadow: 0 0 0 3px rgba(66, 153, 225, 0.3);
}

/* Hide the actual checkbox */
.menu-toggle, .icon-toggle {
    position: absolute;
    opacity: 0;
    width: 0;
    height: 0;
}
    margin: 0.25rem 0;
}
</style>

@endsection
